package rpc

import (
	"errors"
	"time"

	"golang.org/x/net/context"
)

// Client allows calls and notifies on the given transporter, or any protocol
// type. All will share the same ErrorUnwrapper hook for unwrapping incoming
// msgpack objects and converting to possible Go-native `Error` types
type Client struct {
	xp             Transporter
	errorUnwrapper ErrorUnwrapper
	tagsFunc       LogTagsFromContext
	sendNotifier   SendNotifier
}

// NewClient constructs a new client from the given RPC Transporter and the
// ErrorUnwrapper.
func NewClient(xp Transporter, u ErrorUnwrapper,
	tagsFunc LogTagsFromContext) *Client {
	return &Client{xp, u, tagsFunc, nil}
}

// NewClientWithSendNotifier constructs a new client from the given RPC Transporter, the
// ErrorUnwrapper, and the SendNotifier
func NewClientWithSendNotifier(xp Transporter, u ErrorUnwrapper,
	tagsFunc LogTagsFromContext, sendNotifier SendNotifier) *Client {
	return &Client{xp, u, tagsFunc, sendNotifier}
}

// SendNotifier notifies the Caller when an RPC is released into the stream of
// messages. If, for instance, a caller wants to serialize sends to ensure some
// sort of client-side ordering, they can use this hook. Note that the hook fires
// long before the RPC is replied to. It will be called with the RPC sequence number
// that the RPC got on the way out, or SeqNumber(0) for Notify calls (which don't
// get sequence numbers).
type SendNotifier func(SeqNumber)

// Call makes an msgpack RPC call over the transports that's bound to this
// client. The name of the method, and the argument are given. On reply, the
// result field will be populated (if applicable). It returns an Error on
// error, where the error might have been unwrapped from Msgpack via the
// UnwrapErrorFunc in this client. `timeout` will optionally set a deadline on
// the given `ctx`.
func (c *Client) Call(ctx context.Context, method string, arg interface{}, res interface{}, timeout time.Duration) error {
	return c.call(ctx, method, arg, res, CompressionNone, timeout)
}

// CallCompressed acts as Call but allows the response to be compressed with
// the given CompressionType.
func (c *Client) CallCompressed(ctx context.Context, method string,
	arg interface{}, res interface{}, ctype CompressionType, timeout time.Duration) error {
	return c.call(ctx, method, arg, res, ctype, timeout)
}

func (c *Client) call(ctx context.Context, method string,
	arg interface{}, res interface{}, ctype CompressionType, timeout time.Duration) error {
	if ctx == nil {
		return errors.New("No Context provided for this call")
	}
	if timeout > 0 {
		var timeoutCancel context.CancelFunc
		ctx, timeoutCancel = context.WithTimeout(ctx, timeout)
		defer timeoutCancel()
	}

	if c.tagsFunc != nil {
		tags, ok := c.tagsFunc(ctx)
		if ok {
			rpcTags := make(CtxRPCTags)
			for key, tagName := range tags {
				if v := ctx.Value(key); v != nil {
					rpcTags[tagName] = v
				}
			}
			ctx = AddRPCTagsToContext(ctx, rpcTags)
		}
	}

	c.xp.receiveFrames()
	d, err := c.xp.getDispatcher()
	if err != nil {
		return err
	}
	return d.Call(ctx, method, arg, res, ctype, c.errorUnwrapper, c.sendNotifier)
}

// Notify notifies the server, with the given method and argument. It does not
// wait to hear back for an error. An error might happen in sending the call, in
// which case a native Go Error is returned. The UnwrapErrorFunc in the underlying
// client isn't relevant in this case.
func (c *Client) Notify(ctx context.Context, method string, arg interface{}, timeout time.Duration) (err error) {
	if ctx == nil {
		return errors.New("No Context provided for this notification")
	}
	if timeout > 0 {
		var timeoutCancel context.CancelFunc
		ctx, timeoutCancel = context.WithTimeout(ctx, timeout)
		defer timeoutCancel()
	}
	d, err := c.xp.getDispatcher()
	if err != nil {
		return err
	}
	return d.Notify(ctx, method, arg, c.sendNotifier)
}

// GenericClient is the interface that is exported to autogenerated RPC stubs
// from AVDL files.
type GenericClient interface {
	Call(ctx context.Context, method string, arg interface{}, res interface{}, timeout time.Duration) error
	CallCompressed(ctx context.Context, method string,
		arg interface{}, res interface{}, cType CompressionType, timeout time.Duration) error
	Notify(ctx context.Context, method string, arg interface{}, timeout time.Duration) error
}
